<?php

namespace App\Http\Controllers;

use App\Models\Campaign;
use App\Models\EmailList;
use App\Models\EmailTemplate;
use App\Jobs\SendCampaignEmailJob;
use App\Models\CampaignLog;
use Illuminate\Http\Request;

class CampaignController extends Controller
{
    public function index()
    {
        $campaigns = Campaign::with(['list','template'])->latest()->paginate(10);
        return view('campaigns.index', compact('campaigns'));
    }

    public function create()
    {
        $lists = EmailList::all();
        $templates = EmailTemplate::all();
        return view('campaigns.create', compact('lists','templates'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'email_list_id' => 'required|exists:email_lists,id',
            'template_id' => 'required|exists:email_templates,id',
            'scheduled_at' => 'nullable|date',
        ]);
        $campaign = Campaign::create($data + ['status' => 'draft']);
        return redirect()->route('campaigns.index')->with('success', 'Campaign created.');
    }

    public function send(Campaign $campaign)
    {
        $contacts = $campaign->list->contacts()->where('subscribed', true)->get();
        $campaign->update(['status' => 'sending']);

        foreach ($contacts as $contact) {
            CampaignLog::updateOrCreate(
                ['campaign_id' => $campaign->id, 'contact_id' => $contact->id],
                ['status' => 'pending']
            );
            SendCampaignEmailJob::dispatch($campaign, $contact);
        }

        $campaign->update(['status' => 'sent']);
        return back()->with('success','Campaign queued for sending.');
    }

    public function destroy(Campaign $campaign)
    {
        $campaign->delete();
        return back()->with('success', 'Campaign deleted.');
    }
}
