<?php

namespace App\Jobs;

use App\Mail\CampaignMail;
use App\Models\CampaignLog;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;

class SendCampaignEmailJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $campaign;
    public $contact;

    public function __construct($campaign, $contact)
    {
        $this->campaign = $campaign;
        $this->contact = $contact;
    }

    public function handle(): void
    {
        try {
            $template = $this->campaign->template;

            // Personalize the email body
            $body = str_replace(
                ['{{name}}', '{{email}}'],
                [$this->contact->name ?? '', $this->contact->email],
                $template->body
            );

            // Add tracking pixel + tracked links
            $body = $this->injectTracking($body);

            Mail::to($this->contact->email)
                ->send(new CampaignMail($template->subject, $body));

            // Log success
            CampaignLog::updateOrCreate(
                ['campaign_id' => $this->campaign->id, 'contact_id' => $this->contact->id],
                ['status' => 'sent', 'sent_at' => now(), 'error' => null]
            );

        } catch (\Throwable $e) {
            CampaignLog::updateOrCreate(
                ['campaign_id' => $this->campaign->id, 'contact_id' => $this->contact->id],
                ['status' => 'failed', 'error' => $e->getMessage()]
            );
            throw $e;
        }
    }

    private function injectTracking($body)
    {
        $pixelUrl = url("/track/open/{$this->campaign->id}/{$this->contact->id}");
        $pixelTag = "<img src='{$pixelUrl}' width='1' height='1' style='display:none;'>";

        $body = preg_replace_callback(
            '/href=["\'](.*?)["\']/',
            function ($matches) {
                $encoded = urlencode($matches[1]);
                $trackUrl = url("/track/click/{$this->campaign->id}/{$this->contact->id}?url={$encoded}");
                return "href=\"{$trackUrl}\"";
            },
            $body
        );

        return $body . $pixelTag;
    }
}
